<?php

namespace App\Http\Controllers;

use App\Models\JobApplicationAnswer;
use App\Models\JobQuestion;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class JobApplicationAnswerController extends Controller
{
    public function index(): JsonResponse
    {
        $answers = JobApplicationAnswer::with(['application', 'question'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $answers,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $answer = JobApplicationAnswer::with(['application', 'question'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $answer,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Application_id' => 'required|exists:job_applications,Application_id',
            'Question_id' => 'required|exists:job_questions,Question_id',
            'Answer_text' => 'nullable|string',
            'Answer_choice' => 'nullable|array',
            'Answer_choice.*' => 'string',
            'Answer_rating' => 'nullable|integer',
            'Answer_boolean' => 'nullable|boolean',
            'Score' => 'nullable|numeric|between:0,99.99',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $question = JobQuestion::findOrFail($request->input('Question_id'));
        $data = $request->only([
            'Application_id',
            'Question_id',
            'Answer_text',
            'Answer_choice',
            'Answer_rating',
            'Answer_boolean',
            'Score',
        ]);

        // Validate answer based on question type
        $this->validateAnswerByQuestionType($request, $question);

        $answer = JobApplicationAnswer::create($data);

        return response()->json([
            'status' => 'success',
            'data' => $answer->load(['application', 'question']),
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $answer = JobApplicationAnswer::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Application_id' => 'sometimes|exists:job_applications,Application_id',
            'Question_id' => 'sometimes|exists:job_questions,Question_id',
            'Answer_text' => 'nullable|string',
            'Answer_choice' => 'nullable|array',
            'Answer_choice.*' => 'string',
            'Answer_rating' => 'nullable|integer',
            'Answer_boolean' => 'nullable|boolean',
            'Score' => 'nullable|numeric|between:0,99.99',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $question = JobQuestion::findOrFail($request->input('Question_id', $answer->Question_id));
        $data = $request->only([
            'Application_id',
            'Question_id',
            'Answer_text',
            'Answer_choice',
            'Answer_rating',
            'Answer_boolean',
            'Score',
        ]);

        // Validate answer based on question type
        $this->validateAnswerByQuestionType($request, $question);

        $answer->update($data);

        return response()->json([
            'status' => 'success',
            'data' => $answer->load(['application', 'question']),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $answer = JobApplicationAnswer::findOrFail($id);
        $answer->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Job application answer deleted successfully',
        ], 204);
    }

    /**
     * Validate answer based on question type
     */
    private function validateAnswerByQuestionType(Request $request, JobQuestion $question): void
    {
        $errors = [];

        switch ($question->Question_type) {
            case 'text':
                if ($request->has('Answer_text') && is_null($request->input('Answer_text')) && $question->Is_required) {
                    $errors['Answer_text'] = ['The answer text is required for text questions.'];
                }
                if ($request->hasAny(['Answer_choice', 'Answer_rating', 'Answer_boolean'])) {
                    $errors['Answer'] = ['Only Answer_text is allowed for text questions.'];
                }
                if ($question->Max_length && $request->has('Answer_text') && strlen($request->input('Answer_text')) > $question->Max_length) {
                    $errors['Answer_text'] = ["The answer text must not exceed {$question->Max_length} characters."];
                }
                break;

            case 'multiple_choice':
            case 'single_choice':
                if ($request->has('Answer_choice') && is_null($request->input('Answer_choice')) && $question->Is_required) {
                    $errors['Answer_choice'] = ['The answer choice is required for choice questions.'];
                }
                if ($request->has('Answer_choice') && !empty($request->input('Answer_choice'))) {
                    $options = $question->Options ?? [];
                    $invalidChoices = array_diff($request->input('Answer_choice'), $options);
                    if (!empty($invalidChoices)) {
                        $errors['Answer_choice'] = ['Invalid choices provided: ' . implode(', ', $invalidChoices)];
                    }
                    if ($question->Question_type === 'single_choice' && count($request->input('Answer_choice', [])) > 1) {
                        $errors['Answer_choice'] = ['Single choice questions allow only one selection.'];
                    }
                }
                if ($request->hasAny(['Answer_text', 'Answer_rating', 'Answer_boolean'])) {
                    $errors['Answer'] = ['Only Answer_choice is allowed for choice questions.'];
                }
                break;

            case 'rating_scale':
                if ($request->has('Answer_rating') && is_null($request->input('Answer_rating')) && $question->Is_required) {
                    $errors['Answer_rating'] = ['The answer rating is required for rating scale questions.'];
                }
                if ($request->has('Answer_rating') && !is_null($request->input('Answer_rating'))) {
                    $min = $question->Min_value ?? 1;
                    $max = $question->Max_value ?? 10;
                    $rating = $request->input('Answer_rating');
                    if ($rating < $min || $rating > $max) {
                        $errors['Answer_rating'] = ["The answer rating must be between {$min} and {$max}."];
                    }
                }
                if ($request->hasAny(['Answer_text', 'Answer_choice', 'Answer_boolean'])) {
                    $errors['Answer'] = ['Only Answer_rating is allowed for rating scale questions.'];
                }
                break;

            case 'boolean':
                if ($request->has('Answer_boolean') && is_null($request->input('Answer_boolean')) && $question->Is_required) {
                    $errors['Answer_boolean'] = ['The answer boolean is required for boolean questions.'];
                }
                if ($request->hasAny(['Answer_text', 'Answer_choice', 'Answer_rating'])) {
                    $errors['Answer'] = ['Only Answer_boolean is allowed for boolean questions.'];
                }
                break;
        }

        if (!empty($errors)) {
            throw \Illuminate\Validation\ValidationException::withMessages($errors);
        }
    }
}
